#!/bin/bash

# PDF Import Testing Script
# This script demonstrates how to use the PDF import API

BASE_URL="http://localhost:8000"
PDF_FILE="path/to/your/voter_list.pdf"
ADMIN_ID=1

echo "=========================================="
echo "PDF Import API Testing Script"
echo "=========================================="
echo ""

# Colors for output
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Test 1: Analyze PDF Structure
echo -e "${YELLOW}Test 1: Analyzing PDF Structure...${NC}"
echo "Endpoint: POST /api/pdf-import/analyze"
echo ""

if [ -f "$PDF_FILE" ]; then
    ANALYZE_RESPONSE=$(curl -s -X POST "$BASE_URL/api/pdf-import/analyze" \
      -F "pdf_file=@$PDF_FILE")
    
    echo "$ANALYZE_RESPONSE" | jq '.'
    echo ""
else
    echo -e "${RED}Error: PDF file not found at $PDF_FILE${NC}"
    echo "Please update PDF_FILE variable in this script"
    echo ""
fi

# Test 2: Upload PDF for Background Processing
echo -e "${YELLOW}Test 2: Uploading PDF (Background Processing)...${NC}"
echo "Endpoint: POST /api/pdf-import/upload"
echo ""

if [ -f "$PDF_FILE" ]; then
    UPLOAD_RESPONSE=$(curl -s -X POST "$BASE_URL/api/pdf-import/upload" \
      -F "pdf_file=@$PDF_FILE" \
      -F "process_immediately=false" \
      -F "uploaded_by=$ADMIN_ID")
    
    echo "$UPLOAD_RESPONSE" | jq '.'
    
    # Extract import log ID
    IMPORT_ID=$(echo "$UPLOAD_RESPONSE" | jq -r '.data.import_log.id')
    echo ""
    echo -e "${GREEN}Import ID: $IMPORT_ID${NC}"
    echo ""
    
    # Test 3: Check Import Status
    if [ "$IMPORT_ID" != "null" ]; then
        echo -e "${YELLOW}Test 3: Checking Import Status...${NC}"
        echo "Endpoint: GET /api/pdf-import/status/$IMPORT_ID"
        echo ""
        
        sleep 2 # Wait a moment for processing to start
        
        STATUS_RESPONSE=$(curl -s "$BASE_URL/api/pdf-import/status/$IMPORT_ID")
        echo "$STATUS_RESPONSE" | jq '.'
        echo ""
    fi
fi

# Test 4: Get All Imports
echo -e "${YELLOW}Test 4: Getting All Imports...${NC}"
echo "Endpoint: GET /api/pdf-import/all"
echo ""

ALL_IMPORTS=$(curl -s "$BASE_URL/api/pdf-import/all?per_page=5")
echo "$ALL_IMPORTS" | jq '.data.data[] | {id, original_filename, status, total_voters, imported_voters}'
echo ""

# Test 5: Get Statistics
echo -e "${YELLOW}Test 5: Getting Import Statistics...${NC}"
echo "Endpoint: GET /api/pdf-import/statistics"
echo ""

STATS=$(curl -s "$BASE_URL/api/pdf-import/statistics")
echo "$STATS" | jq '{
  total_imports: .data.total_imports,
  pending_imports: .data.pending_imports,
  processing_imports: .data.processing_imports,
  completed_imports: .data.completed_imports,
  failed_imports: .data.failed_imports,
  total_voters_imported: .data.total_voters_imported
}'
echo ""

echo -e "${GREEN}=========================================="
echo "Testing Complete!"
echo "==========================================${NC}"
echo ""
echo "Next Steps:"
echo "1. Check queue worker: php artisan queue:work"
echo "2. Monitor logs: tail -f storage/logs/laravel.log"
echo "3. Import Postman collection: Voter_PDF_Import_API.postman_collection.json"
echo ""
