<?php
namespace App\Services;

use App\Models\Voter;
use App\Models\Booth;
use App\Models\Street;
use App\Repositories\RationCardRepository;
use App\Repositories\VoterRepository;
use App\ResponseHandler\Response;
use App\Validations\Validation;
use Illuminate\Http\Response as LaravelResponse;
use Illuminate\Support\Facades\Log;
use App\Models\Caste;
use App\Models\Category;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class VoterService
{
    protected $rationCardRepository;
    protected $voterRepository;
    protected $response;
    protected $validation;

    public function __construct(
        RationCardRepository $rationCardRepository,
        VoterRepository $voterRepository,
        Response $response,
        Validation $validation
    ) {
        $this->rationCardRepository = $rationCardRepository;
        $this->voterRepository = $voterRepository;
        $this->response = $response;
        $this->validation = $validation;
    }

    /**
     * Create a standalone voter (ration card number is optional)
     * 
     * @param array $data
     * @return \Illuminate\Http\JsonResponse
     */
    public function createVoter(array $data)
    {
        try {
            // Normalize/derive year_of_birth: handle YYYY-MM-DD, 4-digit year, Excel serials, or age
            $currentYear = (int) date('Y');
            if (isset($data['year_of_birth']) && $data['year_of_birth'] !== null && $data['year_of_birth'] !== '') {
                $yRaw = trim((string)$data['year_of_birth']);
                // YYYY-MM-DD
                if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $yRaw)) {
                    $data['year_of_birth'] = (int) substr($yRaw, 0, 4);
                } elseif (is_numeric($yRaw)) {
                    $num = floatval($yRaw);
                    if ($num >= 1900 && $num <= $currentYear) {
                        $data['year_of_birth'] = (int)$num;
                    } elseif ($num > 31) {
                        // Excel serial
                        try {
                            $dt = Date::excelToDateTimeObject($num);
                            $data['year_of_birth'] = (int) $dt->format('Y');
                        } catch (\Exception $e) {
                            // leave as-is; validation will catch
                        }
                    }
                } else {
                    $ts = strtotime($yRaw);
                    if ($ts !== false) {
                        $data['year_of_birth'] = (int) date('Y', $ts);
                    }
                }
            } else {
                // missing explicit year: try date_of_birth or age
                if (!empty($data['date_of_birth']) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $data['date_of_birth'])) {
                    $data['year_of_birth'] = (int)substr($data['date_of_birth'], 0, 4);
                } elseif (!empty($data['age']) && is_numeric($data['age'])) {
                    $data['year_of_birth'] = $currentYear - (int)$data['age'];
                }
            }

            // Step 1: Validate voter data
            $validationResponse = $this->validation->validationForVoter($data);
            $responseData = $validationResponse->getData();

            if ($responseData->status !== 200) {
                return $this->response->respondWithError($responseData->message, $responseData->status);
            }

            // Step 2: Check if ration card exists (if provided)
            $rationCardId = null;
            if (!empty($data['ration_card_number'])) {
                // If this voter is marked as head and an address is provided,
                // ensure the ration card exists (create it if missing).
                if (!empty($data['is_head']) && (int)$data['is_head'] === 1 && !empty($data['ration_card_address'])) {
                    $existingCard = $this->voterRepository->getRationCardByNumber($data['ration_card_number']);
                    if (!$existingCard) {
                        $this->rationCardRepository->create([
                            'ration_card_number' => $data['ration_card_number'],
                            'address' => $data['ration_card_address'],
                            'is_deleted' => false,
                        ]);
                    }
                }

                $rationCard = $this->voterRepository->getRationCardByNumber($data['ration_card_number']);
                if (!$rationCard) {
                    return $this->response->respondWithError(
                        'Ration card number "' . $data['ration_card_number'] . '" does not exist. Please verify the ration card number or create the ration card first.',
                        422
                    );
                }
                $rationCardId = $rationCard->id;
            }

            // Step 3: Handle caste and category: create if not exists, otherwise alert user
            if (!empty($data['caste'])) {
                $casteName = trim($data['caste']);
                // Use existing caste if present, otherwise create it
                $casteModel = Caste::firstOrCreate(['name' => $casteName]);
                $data['caste_id'] = $casteModel->id;
                // keep legacy string for backward compatibility
                $data['caste'] = $casteName;
            }

            if (!empty($data['category'])) {
                $categoryName = trim($data['category']);
                // Use existing category if present, otherwise create it
                $categoryModel = Category::firstOrCreate(['name' => $categoryName]);
                $data['category_id'] = $categoryModel->id;
                // keep legacy string for backward compatibility
                $data['category'] = $categoryName;
            }

            // Step 4: Prepare voter data
            // normalize aadhar for storage
            $normalizedAadhar = null;
            if (isset($data['aadhar_number'])) {
                $aTmp = trim((string)$data['aadhar_number']);
                $digitsTmp = preg_replace('/\D+/', '', $aTmp);
                $normalizedAadhar = $digitsTmp === '' ? null : $digitsTmp;
            }

            $voterData = [
                'ration_card_id' => $rationCardId, // This can be null if no ration card provided
                'voter_id_number' => $data['voter_id_number'],
                'name' => $data['name'],
                'gender' => $data['gender'],
                'year_of_birth' => $data['year_of_birth'] ?? null,
                'date_of_birth' => !empty($data['date_of_birth']) ? $data['date_of_birth'] : null,
                'anniversary_date' => !empty($data['anniversary_date']) ? $data['anniversary_date'] : null,
                'is_head' => isset($data['is_head']) ? (bool)$data['is_head'] : false,
                'booth_id' => !empty($data['booth_id']) ? $data['booth_id'] : null,
                'booth_number' => !empty($data['booth_number']) ? $data['booth_number'] : null,
                'mobile_number' => !empty($data['mobile_number']) ? $data['mobile_number'] : null,
                'aadhar_number' => $normalizedAadhar,
                'street_id' => !empty($data['street_id']) ? $data['street_id'] : null,
                'street_name' => !empty($data['street_name']) ? $data['street_name'] : null,
                'ward_no' => !empty($data['ward_no']) ? $data['ward_no'] : null,
                'caste' => !empty($data['caste']) ? $data['caste'] : null,
                'category' => !empty($data['category']) ? $data['category'] : null,
                'caste_id' => !empty($data['caste_id']) ? $data['caste_id'] : null,
                'category_id' => !empty($data['category_id']) ? $data['category_id'] : null,
                'voter_image' => !empty($data['voter_image']) ? $data['voter_image'] : null,
                'is_deleted' => false, // Explicitly set default value
            ];

            // Step 4: Create voter
            $voter = $this->voterRepository->createVoter($voterData);

            // Step 5: Response
            if (!$voter) {
                return $this->response->respondWithError('Voter ID already exists', 409);
            }

            return $this->response->respondWithData('Voter created successfully', $voter, new LaravelResponse);
        } catch (\Exception $e) {
            // Log the error for debugging
            Log::error('Error creating voter: ' . $e->getMessage(), [
                'data' => $data,
                'trace' => $e->getTraceAsString()
            ]);
            
            return $this->response->respondWithError('Failed to create voter: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get voter by ID
     * 
     * @param int $voterId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getVoterById($voterId)
    {
        $voter = $this->voterRepository->getVoterById($voterId);

        if (!$voter) {
            return $this->response->respondWithError('Voter not found', 404);
        }

        return $this->response->respondWithData('Voter retrieved successfully', $voter, new LaravelResponse);
    }

    /**
     * Update voter
     * 
     * @param int $voterId
     * @param array $data
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateVoter($voterId, array $data)
    {
        try {
            // Normalize/derive year_of_birth before validation (same logic as create)
            $currentYear = (int) date('Y');
            if (isset($data['year_of_birth']) && $data['year_of_birth'] !== null && $data['year_of_birth'] !== '') {
                $yRaw = trim((string)$data['year_of_birth']);
                if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $yRaw)) {
                    $data['year_of_birth'] = (int) substr($yRaw, 0, 4);
                } elseif (is_numeric($yRaw)) {
                    $num = floatval($yRaw);
                    if ($num >= 1900 && $num <= $currentYear) {
                        $data['year_of_birth'] = (int)$num;
                    } elseif ($num > 31) {
                        try {
                            $dt = Date::excelToDateTimeObject($num);
                            $data['year_of_birth'] = (int) $dt->format('Y');
                        } catch (\Exception $e) {
                            // leave as-is
                        }
                    }
                } else {
                    $ts = strtotime($yRaw);
                    if ($ts !== false) {
                        $data['year_of_birth'] = (int) date('Y', $ts);
                    }
                }
            } else {
                if (!empty($data['date_of_birth']) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $data['date_of_birth'])) {
                    $data['year_of_birth'] = (int)substr($data['date_of_birth'], 0, 4);
                } elseif (!empty($data['age']) && is_numeric($data['age'])) {
                    $data['year_of_birth'] = $currentYear - (int)$data['age'];
                }
            }

            // Step 1: Validate voter data
            $validationResponse = $this->validation->validationForVoter($data);
            $responseData = $validationResponse->getData();

            if ($responseData->status !== 200) {
                return $this->response->respondWithError($responseData->message, $responseData->status);
            }

            // Step 2: Check if ration card exists (if provided)
            $rationCardId = null;
            if (!empty($data['ration_card_number'])) {
                // If this voter is marked as head and an address is provided,
                // ensure the ration card exists (create it if missing).
                if (!empty($data['is_head']) && (int)$data['is_head'] === 1 && !empty($data['ration_card_address'])) {
                    $existingCard = $this->voterRepository->getRationCardByNumber($data['ration_card_number']);
                    if (!$existingCard) {
                        $this->rationCardRepository->create([
                            'ration_card_number' => $data['ration_card_number'],
                            'address' => $data['ration_card_address'],
                            'is_deleted' => false,
                        ]);
                    }
                }

                $rationCard = $this->voterRepository->getRationCardByNumber($data['ration_card_number']);
                if (!$rationCard) {
                    return $this->response->respondWithError(
                        'Ration card number "' . $data['ration_card_number'] . '" does not exist. Please verify the ration card number or create the ration card first.',
                        422
                    );
                }
                $rationCardId = $rationCard->id;
            }

            // Step 3: Handle caste and category: create if not exists, otherwise alert user
            if (!empty($data['caste'])) {
                $casteName = trim($data['caste']);
                // Use existing caste if present, otherwise create it
                $casteModel = Caste::firstOrCreate(['name' => $casteName]);
                $data['caste_id'] = $casteModel->id;
                $data['caste'] = $casteName;
            }

            if (!empty($data['category'])) {
                $categoryName = trim($data['category']);
                // Use existing category if present, otherwise create it
                $categoryModel = Category::firstOrCreate(['name' => $categoryName]);
                $data['category_id'] = $categoryModel->id;
                $data['category'] = $categoryName;
            }

            // Step 4: Prepare update data
            // normalize aadhar for storage (update)
            $normalizedAadhar = null;
            if (isset($data['aadhar_number'])) {
                $aTmp = trim((string)$data['aadhar_number']);
                $digitsTmp = preg_replace('/\D+/', '', $aTmp);
                $normalizedAadhar = $digitsTmp === '' ? null : $digitsTmp;
            }

            $updateData = [
                'ration_card_id' => $rationCardId, // This can be null now
                'voter_id_number' => $data['voter_id_number'],
                'name' => $data['name'],
                'gender' => $data['gender'],
                'year_of_birth' => $data['year_of_birth'] ?? null,
                'date_of_birth' => !empty($data['date_of_birth']) ? $data['date_of_birth'] : null,
                'anniversary_date' => !empty($data['anniversary_date']) ? $data['anniversary_date'] : null,
                'is_head' => isset($data['is_head']) ? (bool)$data['is_head'] : false,
                'booth_id' => !empty($data['booth_id']) ? $data['booth_id'] : null,
                'booth_number' => !empty($data['booth_number']) ? $data['booth_number'] : null,
                'mobile_number' => !empty($data['mobile_number']) ? $data['mobile_number'] : null,
                'aadhar_number' => $normalizedAadhar,
                'street_id' => !empty($data['street_id']) ? $data['street_id'] : null,
                'street_name' => !empty($data['street_name']) ? $data['street_name'] : null,
                'ward_no' => !empty($data['ward_no']) ? $data['ward_no'] : null,
                'caste' => !empty($data['caste']) ? $data['caste'] : null,
                'category' => !empty($data['category']) ? $data['category'] : null,
                'caste_id' => !empty($data['caste_id']) ? $data['caste_id'] : null,
                'category_id' => !empty($data['category_id']) ? $data['category_id'] : null,
                'voter_image' => !empty($data['voter_image']) ? $data['voter_image'] : null,
            ];

            // Step 4: Update voter
            $voter = $this->voterRepository->updateVoter($voterId, $updateData);

            if (!$voter) {
                $existingVoter = $this->voterRepository->getVoterById($voterId);
                if (!$existingVoter) {
                    return $this->response->respondWithError('Voter not found', 404);
                } else {
                    return $this->response->respondWithError('Voter ID already exists', 409);
                }
            }

            return $this->response->respondWithData('Voter updated successfully', $voter, new LaravelResponse);
        } catch (\Exception $e) {
            // Log the error for debugging
            Log::error('Error updating voter: ' . $e->getMessage(), [
                'voter_id' => $voterId,
                'data' => $data,
                'trace' => $e->getTraceAsString()
            ]);
            
            return $this->response->respondWithError('Failed to update voter: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Delete voter
     * 
     * @param int $voterId
     * @return \Illuminate\Http\JsonResponse
     */
    public function deleteVoter($voterId)
    {
        $deleted = $this->voterRepository->deleteVoter($voterId);

        if ($deleted) {
            return $this->response->respondWithData('Voter deleted successfully', null, new LaravelResponse);
        }

        return $this->response->respondWithError('Voter not found', 404);
    }

    /**
     * Get all booths for dropdown
     * 
     * @return \Illuminate\Http\JsonResponse
     */
    public function getBoothsForDropdown()
    {
        $booths = Booth::where('is_deleted', false)
                      ->select('id', 'booth_number', 'booth_name')
                      ->orderBy('booth_number')
                      ->get();

        return $this->response->respondWithData('Booths fetched successfully', $booths, new LaravelResponse);
    }

    /**
     * Get all streets for dropdown
     * 
     * @return \Illuminate\Http\JsonResponse
     */
    public function getStreetsForDropdown()
    {
        $streets = Street::where('is_deleted', false)
                        ->select('id', 'street_name')
                        ->orderBy('street_name')
                        ->get();

        return $this->response->respondWithData('Streets fetched successfully', $streets, new LaravelResponse);
    }

    /**
     * Get form data for voter creation (booths + streets)
     * 
     * @return \Illuminate\Http\JsonResponse
     */
    public function getVoterFormData()
    {
        $booths = Booth::where('is_deleted', false)
                      ->select('id', 'booth_number', 'booth_name')
                      ->orderBy('booth_number')
                      ->get();

        $streets = Street::where('is_deleted', false)
                        ->select('id', 'street_name')
                        ->orderBy('street_name')
                        ->get();

        $formData = [
            'booths' => $booths,
            'streets' => $streets,
            'genders' => ['Male', 'Female', 'Other']
        ];

        return $this->response->respondWithData('Voter form data fetched successfully', $formData, new LaravelResponse);
    }

    // Method to get the voter count
    public function getVoterCount()
    {
        return $this->rationCardRepository->getVoterCount();
    }

    /**
     * Import voters from CSV file
     * 
     * @param string $filePath
     * @return \Illuminate\Http\JsonResponse
     */
    public function importVotersFromCsv($filePath)
    {
        try {
            // Read CSV file
            $file = fopen($filePath, 'r');
            if (!$file) {
                return $this->response->respondWithError('Could not open CSV file', 400);
            }

            // Get headers from first row
            $headers = fgetcsv($file);
            if (!$headers) {
                fclose($file);
                return $this->response->respondWithError('CSV file is empty or invalid', 400);
            }
            
            // Normalize headers to lowercase for mapping
            $normalizedHeaders = array_map('strtolower', array_map('trim', $headers));
            
            $successCount = 0;
            $errorCount = 0;
            $errors = [];
            $duplicates = [];
            $rowIndex = 0;

            // Process each row from CSV
            while (($row = fgetcsv($file)) !== false) {
                $rowIndex++;
                $actualRowNumber = $rowIndex + 1; // +1 because header is already read
                
                try {
                    // Skip empty rows
                    if (empty(array_filter($row))) {
                        continue;
                    }

                    // Map row data to voter fields
                    $voterData = $this->mapExcelRowToVoterData($normalizedHeaders, $row);

                    if (empty($voterData['voter_id_number'])) {
                        $errors[] = "Row {$actualRowNumber}: Voter ID Number is required";
                        $errorCount++;
                        continue;
                    }

                    if (empty($voterData['name'])) {
                        $errors[] = "Row {$actualRowNumber}: Name is required";
                        $errorCount++;
                        continue;
                    }

                    // Validate the voter data
                    $validationResponse = $this->validation->validationForVoter($voterData);
                    $responseData = $validationResponse->getData();

                    if ($responseData->status !== 200) {
                        $errors[] = "Row {$actualRowNumber}: " . $responseData->message;
                        $errorCount++;
                        continue;
                    }

                    // Check if ration card exists (if provided)
                    $rationCardId = null;
                    if (!empty($voterData['ration_card_number'])) {
                        // If this voter is head and an address is provided, create the ration card if missing
                        if (!empty($voterData['is_head']) && (int)$voterData['is_head'] === 1 && !empty($voterData['ration_card_address'])) {
                            $existingCard = $this->voterRepository->getRationCardByNumber($voterData['ration_card_number']);
                            if (!$existingCard) {
                                $this->rationCardRepository->create([
                                    'ration_card_number' => $voterData['ration_card_number'],
                                    'address' => $voterData['ration_card_address'],
                                    'is_deleted' => false,
                                ]);
                            }
                        }

                        $rationCard = $this->voterRepository->getRationCardByNumber($voterData['ration_card_number']);
                        if (!$rationCard) {
                            $errors[] = "Row {$actualRowNumber}: Ration card number '{$voterData['ration_card_number']}' does not exist. Please verify the ration card number or create the ration card first.";
                            $errorCount++;
                            continue;
                        }
                        $rationCardId = $rationCard->id;
                    }

                    // Prepare final voter data
                    $finalVoterData = [
                        'ration_card_id' => $rationCardId,
                        'voter_id_number' => $voterData['voter_id_number'],
                        'name' => $voterData['name'],
                        'gender' => $voterData['gender'],
                        'year_of_birth' => $voterData['year_of_birth'] ?? null,
                        'date_of_birth' => !empty($voterData['date_of_birth']) ? $voterData['date_of_birth'] : null,
                        'anniversary_date' => !empty($voterData['anniversary_date']) ? $voterData['anniversary_date'] : null,
                        'is_head' => isset($voterData['is_head']) ? (bool)$voterData['is_head'] : false,
                        'booth_id' => !empty($voterData['booth_id']) ? $voterData['booth_id'] : null,
                        'booth_number' => !empty($voterData['booth_number']) ? $voterData['booth_number'] : null,
                        'mobile_number' => !empty($voterData['mobile_number']) ? $voterData['mobile_number'] : null,
                        'aadhar_number' => !empty($voterData['aadhar_number']) ? $voterData['aadhar_number'] : null,
                        'street_id' => !empty($voterData['street_id']) ? $voterData['street_id'] : null,
                        'street_name' => !empty($voterData['street_name']) ? $voterData['street_name'] : null,
                        'ward_no' => !empty($voterData['ward_no']) ? $voterData['ward_no'] : null,
                        'caste' => !empty($voterData['caste']) ? $voterData['caste'] : null,
                        'category' => !empty($voterData['category']) ? $voterData['category'] : null,
                        'is_deleted' => false,
                    ];

                    // Map caste/category names to ids (use existing or create)
                    if (!empty($finalVoterData['caste'])) {
                        $casteModel = Caste::firstOrCreate(['name' => trim($finalVoterData['caste'])]);
                        $finalVoterData['caste_id'] = $casteModel->id;
                    }

                    if (!empty($finalVoterData['category'])) {
                        $categoryModel = Category::firstOrCreate(['name' => trim($finalVoterData['category'])]);
                        $finalVoterData['category_id'] = $categoryModel->id;
                    }

                    // Map caste/category names to ids (use existing or create)
                    if (!empty($finalVoterData['caste'])) {
                        $casteModel = Caste::firstOrCreate(['name' => trim($finalVoterData['caste'])]);
                        $finalVoterData['caste_id'] = $casteModel->id;
                    }

                    if (!empty($finalVoterData['category'])) {
                        $categoryModel = Category::firstOrCreate(['name' => trim($finalVoterData['category'])]);
                        $finalVoterData['category_id'] = $categoryModel->id;
                    }

                    // Create voter
                    $voter = $this->voterRepository->createVoter($finalVoterData);

                    if (!$voter) {
                        $duplicates[] = "Row {$actualRowNumber}: Voter ID '{$voterData['voter_id_number']}' already exists";
                        $errorCount++;
                    } else {
                        $successCount++;
                    }

                } catch (\Exception $e) {
                    $errors[] = "Row {$actualRowNumber}: " . $e->getMessage();
                    $errorCount++;
                }
            }

            // Prepare response
            $message = "Import completed. {$successCount} voters imported successfully";
            if ($errorCount > 0) {
                $message .= ", {$errorCount} errors occurred";
            }

            $responseData = [
                'success_count' => $successCount,
                'error_count' => $errorCount,
                'total_processed' => $successCount + $errorCount,
                'errors' => array_merge($errors, $duplicates),
            ];

            return $this->response->respondWithData($message, $responseData, new LaravelResponse);

        } catch (\Exception $e) {
            if (isset($file)) {
                fclose($file);
            }
            
            Log::error('Error importing voters from CSV: ' . $e->getMessage(), [
                'file_path' => $filePath,
                'trace' => $e->getTraceAsString()
            ]);
            
            return $this->response->respondWithError('Failed to import CSV file: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Import voters from Excel file (.xlsx, .xls)
     * 
     * @param string $filePath
     * @return \Illuminate\Http\JsonResponse
     */
    public function importVotersFromExcel($filePath)
    {
        try {
            // Check if PhpSpreadsheet is available
            if (!class_exists('\PhpOffice\PhpSpreadsheet\IOFactory')) {
                return $this->response->respondWithError('PhpSpreadsheet library is required for Excel import. Please install it using: composer require phpoffice/phpspreadsheet', 500);
            }

            // Load the Excel file
            $reader = \PhpOffice\PhpSpreadsheet\IOFactory::createReaderForFile($filePath);
            $reader->setReadDataOnly(true); // Only read data, not formatting
            $spreadsheet = $reader->load($filePath);
            $worksheet = $spreadsheet->getActiveSheet();
            
            // Get the highest row and column numbers
            $highestRow = $worksheet->getHighestRow();
            $highestColumn = $worksheet->getHighestColumn();
            $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn);

            if ($highestRow < 2) {
                return $this->response->respondWithError('Excel file must have at least 2 rows (header + data)', 400);
            }

            // Get headers from first row
            $headers = [];
            for ($col = 1; $col <= $highestColumnIndex; $col++) {
                $coordinate = Coordinate::stringFromColumnIndex($col) . '1';
                $cellValue = $worksheet->getCell($coordinate)->getCalculatedValue();
                $headers[] = trim($cellValue);
            }

            if (empty($headers)) {
                return $this->response->respondWithError('Excel file header row is empty', 400);
            }

            // Normalize headers to lowercase for mapping
            $normalizedHeaders = array_map('strtolower', array_map('trim', $headers));
            
            $successCount = 0;
            $errorCount = 0;
            $errors = [];
            $duplicates = [];

            // Process each data row
            for ($row = 2; $row <= $highestRow; $row++) {
                try {
                    // Get row data
                    $rowData = [];
                    $hasData = false;
                    
                    for ($col = 1; $col <= $highestColumnIndex; $col++) {
                        $coordinate = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col) . $row;
                        $cellValue = $worksheet->getCell($coordinate)->getCalculatedValue();
                        
                        // Handle date cells specially
                        $cell = $worksheet->getCell($coordinate);
                        if ($cell->isFormula()) {
                            $cellValue = $cell->getCalculatedValue();
                        } elseif (Date::isDateTime($cell)) {
                            $dateValue = Date::excelToDateTimeObject($cell->getValue());
                            $cellValue = $dateValue->format('Y-m-d');
                        }
                        
                        $rowData[] = $cellValue !== null ? trim((string)$cellValue) : '';
                        if (!empty($cellValue)) {
                            $hasData = true;
                        }
                    }

                    // Skip empty rows
                    if (!$hasData) {
                        continue;
                    }

                    // Map row data to voter fields
                    $voterData = $this->mapExcelRowToVoterData($normalizedHeaders, $rowData);

                    // Normalize year_of_birth for Excel imports: handle YYYY-MM-DD, 4-digit year, Excel serials, or derive from date_of_birth/age
                    $currentYear = (int) date('Y');
                    if (isset($voterData['year_of_birth']) && $voterData['year_of_birth'] !== null && $voterData['year_of_birth'] !== '') {
                        $yRaw = trim((string)$voterData['year_of_birth']);
                        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $yRaw)) {
                            $voterData['year_of_birth'] = (int) substr($yRaw, 0, 4);
                        } elseif (is_numeric($yRaw)) {
                            $num = floatval($yRaw);
                            if ($num >= 1900 && $num <= $currentYear) {
                                $voterData['year_of_birth'] = (int)$num;
                            } elseif ($num > 31) {
                                try {
                                    $dt = Date::excelToDateTimeObject($num);
                                    $voterData['year_of_birth'] = (int) $dt->format('Y');
                                } catch (\Exception $e) {
                                    // leave as-is; validation will catch
                                }
                            }
                        } else {
                            $ts = strtotime($yRaw);
                            if ($ts !== false) {
                                $voterData['year_of_birth'] = (int) date('Y', $ts);
                            }
                        }
                    } else {
                        if (!empty($voterData['date_of_birth']) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $voterData['date_of_birth'])) {
                            $voterData['year_of_birth'] = (int)substr($voterData['date_of_birth'], 0, 4);
                        } elseif (!empty($voterData['age']) && is_numeric($voterData['age'])) {
                            $voterData['year_of_birth'] = $currentYear - (int)$voterData['age'];
                        }
                    }

                    // Normalize aadhar_number: strip non-digits and treat empty as not provided
                    if (isset($voterData['aadhar_number'])) {
                        $aRaw = trim((string)$voterData['aadhar_number']);
                        $aDigits = preg_replace('/\D+/', '', $aRaw);
                        if ($aDigits === '') {
                            unset($voterData['aadhar_number']);
                        } else {
                            $voterData['aadhar_number'] = $aDigits;
                        }
                    }

                    if (empty($voterData['voter_id_number'])) {
                        $errors[] = "Row {$row}: Voter ID Number is required";
                        $errorCount++;
                        continue;
                    }

                    if (empty($voterData['name'])) {
                        $errors[] = "Row {$row}: Name is required";
                        $errorCount++;
                        continue;
                    }

                    // Validate the voter data
                    $validationResponse = $this->validation->validationForVoter($voterData);
                    $responseData = $validationResponse->getData();

                    if ($responseData->status !== 200) {
                        $errors[] = "Row {$row}: " . $responseData->message;
                        $errorCount++;
                        continue;
                    }

                    // Check if ration card exists (if provided)
                    $rationCardId = null;
                    if (!empty($voterData['ration_card_number'])) {
                        // If this voter is head and an address is provided, create the ration card if missing
                        if (!empty($voterData['is_head']) && (int)$voterData['is_head'] === 1 && !empty($voterData['ration_card_address'])) {
                            $existingCard = $this->voterRepository->getRationCardByNumber($voterData['ration_card_number']);
                            if (!$existingCard) {
                                $this->rationCardRepository->create([
                                    'ration_card_number' => $voterData['ration_card_number'],
                                    'address' => $voterData['ration_card_address'],
                                    'is_deleted' => false,
                                ]);
                            }
                        }

                        $rationCard = $this->voterRepository->getRationCardByNumber($voterData['ration_card_number']);
                        if (!$rationCard) {
                            $errors[] = "Row {$row}: Ration card number '{$voterData['ration_card_number']}' does not exist. Please verify the ration card number or create the ration card first.";
                            $errorCount++;
                            continue;
                        }
                        $rationCardId = $rationCard->id;
                    }

                    // Prepare final voter data
                    $finalVoterData = [
                        'ration_card_id' => $rationCardId,
                        'voter_id_number' => $voterData['voter_id_number'],
                        'name' => $voterData['name'],
                        'gender' => $voterData['gender'],
                        'year_of_birth' => $voterData['year_of_birth'] ?? null,
                        'date_of_birth' => !empty($voterData['date_of_birth']) ? $voterData['date_of_birth'] : null,
                        'anniversary_date' => !empty($voterData['anniversary_date']) ? $voterData['anniversary_date'] : null,
                        'is_head' => isset($voterData['is_head']) ? (bool)$voterData['is_head'] : false,
                        'booth_id' => !empty($voterData['booth_id']) ? $voterData['booth_id'] : null,
                        'booth_number' => !empty($voterData['booth_number']) ? $voterData['booth_number'] : null,
                        'mobile_number' => !empty($voterData['mobile_number']) ? $voterData['mobile_number'] : null,
                        'aadhar_number' => !empty($voterData['aadhar_number']) ? $voterData['aadhar_number'] : null,
                        'street_id' => !empty($voterData['street_id']) ? $voterData['street_id'] : null,
                        'street_name' => !empty($voterData['street_name']) ? $voterData['street_name'] : null,
                        'ward_no' => !empty($voterData['ward_no']) ? $voterData['ward_no'] : null,
                        'caste' => !empty($voterData['caste']) ? $voterData['caste'] : null,
                        'category' => !empty($voterData['category']) ? $voterData['category'] : null,
                        'is_deleted' => false,
                    ];

                    // Create voter
                    $voter = $this->voterRepository->createVoter($finalVoterData);

                    if (!$voter) {
                        $duplicates[] = "Row {$row}: Voter ID '{$voterData['voter_id_number']}' already exists";
                        $errorCount++;
                    } else {
                        $successCount++;
                    }

                } catch (\Exception $e) {
                    $errors[] = "Row {$row}: " . $e->getMessage();
                    $errorCount++;
                }
            }

            // Clean up
            $spreadsheet->disconnectWorksheets();
            unset($spreadsheet);

            // Prepare response
            $message = "Excel import completed. {$successCount} voters imported successfully";
            if ($errorCount > 0) {
                $message .= ", {$errorCount} errors occurred";
            }

            $responseData = [
                'success_count' => $successCount,
                'error_count' => $errorCount,
                'total_processed' => $successCount + $errorCount,
                'errors' => array_merge($errors, $duplicates),
            ];

            return $this->response->respondWithData($message, $responseData, new LaravelResponse);

        } catch (\Exception $e) {
            Log::error('Error importing voters from Excel: ' . $e->getMessage(), [
                'file_path' => $filePath,
                'trace' => $e->getTraceAsString()
            ]);
            
            return $this->response->respondWithError('Failed to import Excel file: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Generate sample Excel template for voter import
     * 
     * @return \Illuminate\Http\Response
     */
    public function generateExcelTemplate()
    {
        try {
            // Check if PhpSpreadsheet is available
            if (!class_exists('\PhpOffice\PhpSpreadsheet\Spreadsheet')) {
                return response()->json(['error' => 'PhpSpreadsheet library is required. Please install it using: composer require phpoffice/phpspreadsheet'], 500);
            }

            // Create new spreadsheet
            $spreadsheet = new Spreadsheet();
            $sheet = $spreadsheet->getActiveSheet();
            $sheet->setTitle('Voter Import Template');

            // Define headers
            $headers = [
                'voter_id_number',
                'name',
                'gender',
                'year_of_birth',
                'date_of_birth',
                'anniversary_date',
                'is_head',
                'booth_id',
                'booth_number',
                'mobile_number',
                'aadhar_number',
                'street_id',
                'street_name',
                'ward_no',
                'caste',
                'category',
                'ration_card_address',
                'ration_card_number'
            ];

            // Set headers with styling
            $col = 'A';
            foreach ($headers as $header) {
                $sheet->setCellValue($col . '1', $header);
                $sheet->getStyle($col . '1')->getFont()->setBold(true);
                $sheet->getStyle($col . '1')->getFill()
                    ->setFillType(Fill::FILL_SOLID)
                    ->getStartColor()->setARGB('FFE6E6FA');
                $sheet->getColumnDimension($col)->setAutoSize(true);
                $col++;
            }

            // Sample data
            $sampleData = [
                ['BCC1234567', 'RAJESWARI', 'Female', '1989', '1989-06-09', '2017-06-26', '0', '18', 'B003', '9698979194', '348234234234', '10', 'Mettu street vikravandi', 'Ward 1', 'BC', 'General', '', ''],
                ['BCC1234568', 'KUMAR', 'Male', '1985', '1985-03-15', '', '1', '19', 'B004', '9876543210', '123456789012', '11', 'Main street', 'Ward 2', 'SC', 'Reserved', 'RC123456', '123 Main Street'],
                ['BCC1234569', 'PRIYA', 'Female', '1992', '1992-12-20', '2015-04-10', '1', '20', 'B005', '9876543211', '234567890123', '12', 'Church street', 'Ward 3', 'OBC', 'General', 'RC123457', '45 Church Road']
            ];

            // Add sample data
            $row = 2;
            foreach ($sampleData as $data) {
                $col = 'A';
                foreach ($data as $value) {
                    $sheet->setCellValue($col . $row, $value);
                    $col++;
                }
                $row++;
            }

            // Create temporary file
            $tempFile = tempnam(sys_get_temp_dir(), 'voter_template_');
            $writer = new Xlsx($spreadsheet);
            $writer->save($tempFile);

            // Clean up
            $spreadsheet->disconnectWorksheets();
            unset($spreadsheet);

            return response()->download($tempFile, 'voter_import_template.xlsx', [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            ])->deleteFileAfterSend(true);

        } catch (\Exception $e) {
            Log::error('Error generating Excel template: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to generate Excel template'], 500);
        }
    }

    /**
     * Map Excel row data to voter data array
     * 
     * @param array $headers
     * @param array $row
     * @return array
     */
    private function mapExcelRowToVoterData($headers, $row)
    {
        $mapping = [
            'voter_id_number' => ['voter_id_number', 'voter id number', 'voter_id', 'id_number', 'voter id'],
            'name' => ['name', 'voter_name', 'full_name', 'voter name'],
            'gender' => ['gender', 'sex'],
            'year_of_birth' => ['year_of_birth', 'birth_year', 'yob', 'year of birth'],
            'date_of_birth' => ['date_of_birth', 'dob', 'birth_date', 'date of birth'],
            'anniversary_date' => ['anniversary_date', 'anniversary', 'marriage_date'],
            'is_head' => ['is_head', 'head_of_family', 'family_head', 'head'],
            'booth_id' => ['booth_id', 'booth id'],
            'booth_number' => ['booth_number', 'booth number', 'booth_no', 'booth'],
            'mobile_number' => ['mobile_number', 'mobile', 'phone', 'contact', 'mobile number'],
            'aadhar_number' => ['aadhar_number', 'aadhar', 'aadhaar', 'aadhar number'],
            'street_id' => ['street_id', 'street id'],
            'street_name' => ['street_name', 'street', 'address', 'street name'],
            'ward_no' => ['ward_no', 'ward', 'ward_number', 'ward number'],
            'caste' => ['caste', 'community', 'jati'],
            'category' => ['category', 'reservation', 'quota', 'type'],
            'ration_card_number' => ['ration_card_number', 'ration card', 'ration_card', 'card_number'],
            'ration_card_address' => ['ration_card_address', 'ration card address', 'ration_address', 'address']
        ];

        $voterData = [];

        foreach ($mapping as $field => $possibleHeaders) {
            foreach ($possibleHeaders as $possibleHeader) {
                $index = array_search($possibleHeader, $headers);
                if ($index !== false && isset($row[$index])) {
                    $value = trim($row[$index]);
                    
                    // Handle special cases
                    if ($field === 'is_head') {
                        $voterData[$field] = in_array(strtolower($value), ['1', 'true', 'yes', 'y']) ? 1 : 0;
                    } elseif ($field === 'gender' && !empty($value)) {
                        // Normalize gender values
                        $gender = ucfirst(strtolower($value));
                        if (in_array($gender, ['M', 'Male', 'man'])) {
                            $voterData[$field] = 'Male';
                        } elseif (in_array($gender, ['F', 'Female', 'woman'])) {
                            $voterData[$field] = 'Female';
                        } else {
                            $voterData[$field] = 'Other';
                        }
                    } elseif (in_array($field, ['date_of_birth', 'anniversary_date']) && !empty($value)) {
                        // Handle date formatting
                        try {
                            // Try to parse various date formats
                            $date = new \DateTime($value);
                            $voterData[$field] = $date->format('Y-m-d');
                        } catch (\Exception $e) {
                            // Try common date formats
                            $formats = ['d/m/Y', 'd-m-Y', 'm/d/Y', 'm-d-Y', 'Y/m/d'];
                            foreach ($formats as $format) {
                                $date = \DateTime::createFromFormat($format, $value);
                                if ($date && $date->format($format) === $value) {
                                    $voterData[$field] = $date->format('Y-m-d');
                                    break;
                                }
                            }
                            if (!isset($voterData[$field])) {
                                $voterData[$field] = null;
                            }
                        }
                    } elseif (!empty($value)) {
                        $voterData[$field] = $value;
                    }
                    break; // Found a match, move to next field
                }
            }
        }

        return $voterData;
    }

    /**
     * Generate sample CSV template for voter import
     * 
     * @return \Illuminate\Http\Response
     */
    public function generateCsvTemplate()
    {
        try {
            // Define headers
            $headers = [
                'voter_id_number',
                'name',
                'gender',
                'year_of_birth',
                'date_of_birth',
                'anniversary_date',
                'is_head',
                'booth_id',
                'booth_number',
                'mobile_number',
                'aadhar_number',
                'street_id',
                'street_name',
                'ward_no',
                'caste',
                'category',
                'ration_card_address',
                'ration_card_number'
            ];

            // Sample data
            $sampleData = [
                ['BCC1234567', 'RAJESWARI', 'Female', '1989', '1989-06-09', '2017-06-26', '0', '18', 'B003', '9698979194', '348234234234', '10', 'Mettu street, vikravandi', 'Ward 1', 'BC', 'General', '', ''],
                ['BCC1234568', 'KUMAR', 'Male', '1985', '1985-03-15', '', '1', '19', 'B004', '9876543210', '123456789012', '11', 'Main street', 'Ward 2', 'SC', 'Reserved', 'RC123456', '123 Main Street']
            ];

            // Create CSV content
            $tempFile = tempnam(sys_get_temp_dir(), 'voter_template_');
            $file = fopen($tempFile, 'w');

            // Write headers
            fputcsv($file, $headers);

            // Write sample data
            foreach ($sampleData as $row) {
                fputcsv($file, $row);
            }

            fclose($file);

            return response()->download($tempFile, 'voter_import_template.csv', [
                'Content-Type' => 'text/csv',
            ])->deleteFileAfterSend(true);

        } catch (\Exception $e) {
            Log::error('Error generating CSV template: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to generate template'], 500);
        }
    }

    /**
     * Map existing voter to ration card
     * 
     * @param array $data
     * @return \Illuminate\Http\JsonResponse
     */
    public function mapVoterToRationCard(array $data)
    {
        try {
            // Validate required fields
            if (empty($data['voter_id_number'])) {
                return $this->response->respondWithError('Voter ID number is required', 400);
            }

            if (empty($data['ration_card_number'])) {
                return $this->response->respondWithError('Ration card number is required', 400);
            }

            // Check if ration card exists
            $rationCard = $this->voterRepository->getRationCardByNumber($data['ration_card_number']);
            if (!$rationCard) {
                return $this->response->respondWithError('Ration card not found', 404);
            }

            // Map voter to ration card using repository method
            $result = $this->voterRepository->mapVoterToRationCard($data['voter_id_number'], $rationCard->id);

            if (!$result['success']) {
                // Use 404 status code specifically for "not found" errors
                $statusCode = strpos($result['message'], 'not found') !== false ? 404 : 400;
                return $this->response->respondWithError($result['message'], $statusCode);
            }

            // Load voter with relations for response
            $voter = $this->voterRepository->getByVoterIdNumber($data['voter_id_number']);
            $voter->load(['rationCard', 'street', 'booth']);

            return $this->response->respondWithData($result['message'], $voter, new LaravelResponse);

        } catch (\Exception $e) {
            Log::error('Error mapping voter to ration card: ' . $e->getMessage(), [
                'data' => $data,
                'trace' => $e->getTraceAsString()
            ]);
            
            return $this->response->respondWithError('Failed to map voter to ration card: ' . $e->getMessage(), 500);
        }
    }
}