<?php

namespace App\Services;

use App\Repositories\StreetRepository;
use App\ResponseHandler\Response;
use Illuminate\Http\Response as LaravelResponse;
use App\Validations\Validation;

class StreetService
{
    protected $streetRepository;
    protected $response;
    protected $validation;

    public function __construct(StreetRepository $streetRepository, Response $response, Validation $validation)
    {
        $this->streetRepository = $streetRepository;
        $this->response = $response;
        $this->validation = $validation;
    }

    // Create Street
    public function createStreet($data)
    {
        // Step 1: Validate street data
        $validationResponse = $this->validation->validationForStreet($data);
        $responseData = $validationResponse->getData();

        if ($responseData->status !== 200) {
            return $this->response->respondWithError($responseData->message, $responseData->status);
        }

        // Step 2: Create street
        $street = $this->streetRepository->createStreet([
            'street_name' => $data['street_name']
        ]);

        // Step 3: Response
        if (!$street) {
            return $this->response->respondWithError('Street name already exists', 409);
        }

        return $this->response->respondWithData('Street created successfully', $street, new LaravelResponse);
    }

    // Get all Streets
    public function getAllStreets()
    {
        $streets = $this->streetRepository->getAll();
        $streetCount = $this->streetRepository->getStreetCount();

        // Add voters_count to each street
        $streetsWithVotersCount = $streets->map(function ($street) {
            $votersCount = \App\Models\Voter::where('street_id', $street->id)
                                           ->where('is_deleted', false)
                                           ->count();
            
            // Convert street to array and add voters_count
            $streetArray = $street->toArray();
            $streetArray['voters_count'] = $votersCount;
            
            return $streetArray;
        });

        return response()->json([
            'status' => 200,
            'message' => 'Streets fetched successfully',
            'data' => $streetsWithVotersCount,
            'count' => $streetCount,
        ]);
    }

    // Update Street
    public function updateStreet($streetId, array $data)
    {
        // Step 1: Validate street data
        $validationResponse = $this->validation->validationForStreet($data);
        $responseData = $validationResponse->getData();

        if ($responseData->status !== 200) {
            return $this->response->respondWithError($responseData->message, $responseData->status);
        }

        // Step 2: Update street
        $street = $this->streetRepository->update($streetId, [
            'street_name' => $data['street_name']
        ]);

        if (!$street) {
            // Check if street exists to provide more specific error
            $existingStreet = $this->streetRepository->getStreetById($streetId);
            if (!$existingStreet) {
                return $this->response->respondWithError('Street not found', 404);
            } else {
                return $this->response->respondWithError('Street name already exists', 409);
            }
        }

        // Step 3: Response
        return $this->response->respondWithData('Street updated successfully', $street, new LaravelResponse());
    }

    // Soft Delete Street
    public function deleteStreet($streetId)
    {
        if (empty($streetId)) {
            return $this->response->respondWithError('Street ID is required');
        }

        $deleted = $this->streetRepository->delete($streetId);

        if ($deleted) {
            return $this->response->respondWithData('Street deleted successfully', null, new LaravelResponse());
        }

        return $this->response->respondWithError('Street not found or already deleted', 404);
    }
}
