<?php

namespace App\Jobs;

use App\Models\ImagePageImportLog;
use App\Models\Voter;
use App\Models\Booth;
use App\Models\RationCard;
use App\Services\Concerns\ParsesVoterBlocks;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use thiagoalessio\TesseractOCR\TesseractOCR;
use Exception;

class ProcessVoterImagePage implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels, ParsesVoterBlocks;

    public $tries = 2;
    public $timeout = 600; // 10 minutes per page

    public string $pagePath; // Made public for ParsesVoterBlocks trait access
    protected ?string $boothNumber;
    protected int $yearContext;
    protected ImagePageImportLog $log;

    public function __construct(ImagePageImportLog $log, string $pagePath, ?string $boothNumber = null, ?int $yearContext = null)
    {
        $this->log = $log;
        $this->pagePath = $pagePath;
        $this->boothNumber = $boothNumber;
        $this->yearContext = $yearContext ?? (int)date('Y');
    }

    public function handle(): void
    {
        $this->log->markProcessing();
        Log::info('Processing voter image page', ['page' => $this->pagePath]);

        if (!file_exists($this->pagePath)) {
            $this->log->markFailed(['File missing']);
            return;
        }

        try {
            $ocr = new TesseractOCR($this->pagePath);
            $ocr->lang('eng');
            $ocr->psm(6); // Assume uniform text blocks
            $text = $ocr->run();

            $voters = $this->parseVotersFromOcr($text, $this->boothNumber, $this->pagePath);
            $found = count($voters);
            $imported = 0; $updated = 0; $failed = 0; $errors = [];

            // Per requirements ration_card_id should remain null for imported image voters
            $rationCardId = $this->getRationCardId();
            $boothId = $this->ensureBooth($this->boothNumber);
            
            // Extract section and street information from header
            $headerInfo = $this->extractSectionAndStreetFromHeader($this->pagePath);
            $streetInfo = $this->ensureStreetExists($headerInfo['street_name'], $boothId);
            
            // Track used serial numbers within this booth to ensure uniqueness
            $usedSerialNumbers = $this->getExistingSerialNumbers($boothId);
            $nextAvailableSerial = $this->getNextAvailableSerial($usedSerialNumbers);
            
            // Assign unique serial numbers to voters
            $voters = $this->assignUniqueSerialNumbers($voters, $usedSerialNumbers, $nextAvailableSerial);

            foreach ($voters as $voterData) {
                try {
                    $existing = Voter::where('voter_id_number', $voterData['voter_id_number'])->first();
                    $payload = [
                        'serial_number' => $voterData['serial_no'] ?? null,
                        'ration_card_id' => $rationCardId,
                        'voter_id_number' => $voterData['voter_id_number'],
                        'name' => $voterData['name'],
                        'relation_type' => $voterData['relation_type'] ?? null,
                        'relation_name' => $voterData['relation_name'] ?? null,
                        'house_number' => $voterData['house_number'] ?? null,
                        'age' => $voterData['age'] ?? null,
                        'gender' => $voterData['gender'] ?? 'Male', // Add fallback for gender
                        'year_of_birth' => $voterData['year_of_birth'],
                        'booth_number' => $voterData['booth_number'],
                        'booth_id' => $boothId,
                        'street_id' => $streetInfo['street_id'],
                        'street_name' => $streetInfo['street_name'],
                        'is_head' => false,
                        'is_deleted' => false,
                    ];
                    if ($voterData['is_deleted'] ?? false) {
                        if ($existing) {
                            $existing->update(['is_deleted' => true]);
                        }
                        // Skip creation of deleted voter not present
                    } else {
                        $payload['is_deleted'] = false;
                        if ($existing) {
                            $existing->update($payload);
                            $updated++;
                        } else {
                            Voter::create($payload);
                            $imported++;
                        }
                    }
                } catch (Exception $e) {
                    $failed++; $errors[] = $e->getMessage();
                    Log::warning('Failed inserting voter from image', ['epic' => $voterData['voter_id_number'] ?? 'unknown', 'error' => $e->getMessage()]);
                }
            }

            if ($failed > 0) {
                $this->log->update(['errors' => $errors]);
            }
            $this->log->markCompleted($found, $imported, $failed);
            Log::info('Completed image page import', [
                'page' => $this->pagePath, 
                'found' => $found, 
                'new_imported' => $imported,
                'updated' => $updated,
                'total_processed' => $imported + $updated,
                'failed' => $failed
            ]);
        } catch (Exception $e) {
            $this->log->markFailed([$e->getMessage()]);
            Log::error('Image page import failed', ['page' => $this->pagePath, 'error' => $e->getMessage()]);
            throw $e; // Let queue handle retry
        }
    }

    // Requirement: ration_card_id should be null for these imported voters
    protected function getRationCardId(): ?int
    {
        return null;
    }

    protected function ensureBooth(?string $boothNumber): ?int
    {
        if (!$boothNumber) return null;
        $booth = Booth::firstOrCreate(
            ['booth_number' => $boothNumber],
            ['booth_address' => 'Unknown', 'is_deleted' => false]
        );
        return $booth->id;
    }

    /**
     * Extract section number and street name from image header
     * Expected format: "Section No and Name : 1-MURUGAN KOIL STREET, MANALIPET, Puducherry - 605501"
     */
    private function extractSectionAndStreetFromHeader(string $imagePath): array
    {
        try {
            $ocr = new TesseractOCR($imagePath);
            $ocr->lang('eng')->psm(6);
            $text = $ocr->run();
            
            $sectionNumber = null;
            $streetName = null;
            
            // Look for "Section No and Name" pattern
            if (preg_match('/Section\\s+No\\s+and\\s+Name\\s*[:\\s]\\s*(\\d+)[\\-\\s]*(.+?)$/im', $text, $match)) {
                $sectionNumber = trim($match[1]);
                $fullAddress = trim($match[2]);
                
                // Extract street name (everything before city/state/pincode)
                // Remove common suffixes like city, state, pincode
                $streetName = preg_replace('/,\\s*(?:Puducherry|Tamil Nadu|Kerala|Karnataka|Andhra Pradesh|Telangana)\\s*[\\-\\s]*\\d{6}.*$/i', '', $fullAddress);
                $streetName = preg_replace('/,\\s*\\d{6}.*$/', '', $streetName); // Remove pincode at end
                $streetName = trim($streetName, ' ,-');
            }
            
            Log::info('Extracted section info from header', [
                'section_number' => $sectionNumber,
                'street_name' => $streetName,
                'image' => basename($imagePath)
            ]);
            
            return [
                'section_number' => $sectionNumber,
                'street_name' => $streetName
            ];
        } catch (\Exception $e) {
            Log::error('Failed to extract section info from header', [
                'image' => $imagePath, 
                'error' => $e->getMessage()
            ]);
            return [
                'section_number' => null,
                'street_name' => null
            ];
        }
    }

    /**
     * Ensure street exists and return street_id and street_name
     */
    private function ensureStreetExists(?string $streetName, ?int $boothId): array
    {
        if (empty($streetName) || empty($boothId)) {
            return ['street_id' => null, 'street_name' => null];
        }
        
        // Check if street already exists with this booth_id
        $street = \App\Models\Street::where('street_name', $streetName)
                                    ->where('booth_id', $boothId)
                                    ->where('is_deleted', false)
                                    ->first();
        
        if (!$street) {
            // Create new street record
            $street = \App\Models\Street::create([
                'street_name' => $streetName,
                'booth_id' => $boothId,
                'is_deleted' => false
            ]);
            
            Log::info('Created new street record', [
                'street_id' => $street->id,
                'street_name' => $streetName,
                'booth_id' => $boothId
            ]);
        }
        
        return [
            'street_id' => $street->id,
            'street_name' => $street->street_name
        ];
    }
    
    /**
     * Get existing serial numbers for a booth to avoid duplicates
     */
    private function getExistingSerialNumbers(?int $boothId): array
    {
        if (!$boothId) {
            return [];
        }
        
        return Voter::where('booth_id', $boothId)
                   ->where('is_deleted', false)
                   ->whereNotNull('serial_number')
                   ->pluck('serial_number')
                   ->toArray();
    }
    
    /**
     * Get next available serial number
     */
    private function getNextAvailableSerial(array $usedSerials): int
    {
        $nextSerial = 1;
        while (in_array($nextSerial, $usedSerials)) {
            $nextSerial++;
        }
        return $nextSerial;
    }
    
    /**
     * Assign unique serial numbers to voters, preserving OCR-extracted serials where possible
     */
    private function assignUniqueSerialNumbers(array $voters, array &$usedSerials, int &$nextAvailable): array
    {
        foreach ($voters as &$voter) {
            $ocrSerial = $voter['serial_no'] ?? null;
            
            // If OCR extracted a valid serial and it's not used, keep it
            if ($ocrSerial && is_numeric($ocrSerial) && $ocrSerial > 0 && !in_array((int)$ocrSerial, $usedSerials)) {
                $finalSerial = (int)$ocrSerial;
                $usedSerials[] = $finalSerial;
                $voter['serial_no'] = $finalSerial;
                
                Log::debug('Using OCR-extracted serial number', [
                    'voter_id' => $voter['voter_id_number'] ?? 'unknown',
                    'serial' => $finalSerial
                ]);
            } else {
                // OCR serial is invalid/duplicate, assign next available
                while (in_array($nextAvailable, $usedSerials)) {
                    $nextAvailable++;
                }
                
                $usedSerials[] = $nextAvailable;
                $voter['serial_no'] = $nextAvailable;
                
                Log::info('Assigned new unique serial number', [
                    'voter_id' => $voter['voter_id_number'] ?? 'unknown',
                    'ocr_serial' => $ocrSerial,
                    'assigned_serial' => $nextAvailable,
                    'reason' => $ocrSerial ? 'OCR serial already used' : 'No OCR serial found'
                ]);
                
                $nextAvailable++;
            }
        }
        
        return $voters;
    }
}
