<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Jobs\ProcessVoterImageBatch;
use Illuminate\Support\Facades\Log;
use App\Security\DecryptUtils;

class VoterImageBatchController extends Controller
{
    /**
     * Get default crop options for image processing
     */
    private function getDefaultCropOptions(): array
    {
        return [
            'cols' => 3,
            'rows' => 10,
            'x_offset' => 0,
            'y_offset' => 75,   // Skip header, start at first voter card
            'bottom_offset' => 100, // Reserve space for footer but not too much
            'pad_x' => 0,       // No horizontal padding
            'pad_y' => 6,       // Default vertical padding
            // Row 10: reduce padding and carefully extend to capture voter IDs without footer
            'row_pad_y' => [10 => 2],
            // Extend row 10 slightly to capture voter IDs, but stay clear of footer (~50px)
            'row_extend_bottom' => [10 => 50],
            // Enable dynamic detection of voter boxes when true; overrides offsets & box size.
            'dynamic' => false
        ];
    }
    
    /**
     * Prepare import options from parameters
     */
    private function prepareImportOptions(array $params): array
    {
        $defaultCropOptions = $this->getDefaultCropOptions();
        
        // Merge request crop_options with defaults (request params override defaults)
        $requestCropOptions = $params['crop_options'] ?? [];
        $mergedCropOptions = array_merge($defaultCropOptions, $requestCropOptions);
        
        // Handle dynamic parameter from both root level and crop_options
        if (isset($params['dynamic'])) {
            $mergedCropOptions['dynamic'] = filter_var($params['dynamic'], FILTER_VALIDATE_BOOLEAN);
        } elseif (isset($requestCropOptions['dynamic'])) {
            $mergedCropOptions['dynamic'] = filter_var($requestCropOptions['dynamic'], FILTER_VALIDATE_BOOLEAN);
        }
        
        return [
            'use_crops' => $params['use_crops'] ?? true, // Default to true
            'keep_crops' => $params['keep_crops'] ?? false,
            'reset_booth_voters' => $params['reset_booth_voters'] ?? true,
            'crop_options' => $mergedCropOptions
        ];
    }
    
    /**
     * HTTP endpoint for running batch import (handles both encrypted and plain JSON requests)
     */
    public function run(Request $request)
    {
        // Try to get JSON data first (plain JSON request)
        $params = $request->json()->all();
        
        // If no JSON data, try decrypting (encrypted request)
        if (empty($params)) {
            $data = $request->getContent();
            $params = DecryptUtils::decryptRequestData($data);
        }
        
        // Validate we have data
        if (empty($params) || !is_array($params)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid request data',
                'error' => 'Request must contain valid JSON or encrypted data'
            ], 400);
        }
        
        // Call the actual import logic with decrypted data
        return $this->runImport($params);
    }
    
    /**
     * Internal method for running batch import (accepts plain array)
     * Can be called from other controllers without encryption
     */
    public function runImport(array $params)
    {
        // New hierarchical structure: Constituency/[ConstituencyName]/[BoothNumber]
        $constituencyName = $params['constituency'] ?? null;
        $boothNumber = $params['booth_number'] ?? null;
        
        // Validate required parameters
        if (!$constituencyName) {
            return response()->json([
                'success' => false,
                'message' => 'Constituency name is required',
                'error' => 'Please provide constituency parameter'
            ], 400);
        }
        
        if (!$boothNumber) {
            return response()->json([
                'success' => false,
                'message' => 'Booth number is required',
                'error' => 'Please provide booth_number parameter'
            ], 400);
        }
        
        // Construct directory path: Constituency/[ConstituencyName]/[BoothNumber]
        // Use parent-level Constituency folder (shared across all client subdomains)
        $constituencyFolder = config('filesystems.disks.constituency.root');
        $constituencyPath = $constituencyFolder . '/' . $constituencyName;
        $directory = $constituencyPath . '/' . $boothNumber;
        
        // Validate constituency folder exists
        if (!is_dir($constituencyPath)) {
            return response()->json([
                'success' => false,
                'message' => 'Constituency folder not found',
                'error' => "Constituency '{$constituencyName}' does not exist under Constituency folder",
                'path_checked' => $constituencyPath
            ], 404);
        }
        
        // Validate booth folder exists
        if (!is_dir($directory)) {
            return response()->json([
                'success' => false,
                'message' => 'Booth folder not found',
                'error' => "Booth '{$boothNumber}' does not exist under {$constituencyName}",
                'path_checked' => $directory
            ], 404);
        }
        
        $startIndex = (int)($params['start_index'] ?? 0); // 0-based index of first voter page
        $maxImages = isset($params['max_images']) ? (int)$params['max_images'] : null;
        
        // Prepare import options using shared method
        $options = $this->prepareImportOptions($params);

        // Execute synchronously to get detailed results
        $job = new ProcessVoterImageBatch($directory, $startIndex, $maxImages, $options);
        $results = $job->handle();
        
        Log::info('Completed voter image batch import', ['directory' => $directory, 'results' => $results]);

        // Generate download URLs for Excel files if they exist
        $skippedExcelUrl = null;
        $warningExcelUrl = null;
        
        if (!empty($results['skipped_excel_path']) && file_exists($results['skipped_excel_path'])) {
            $skippedExcelUrl = route('constituency.file', [
                'constituency' => $constituencyName,
                'booth' => $boothNumber,
                'filename' => basename($results['skipped_excel_path'])
            ]);
        }
        
        if (!empty($results['warning_excel_path']) && file_exists($results['warning_excel_path'])) {
            $warningExcelUrl = route('constituency.file', [
                'constituency' => $constituencyName,
                'booth' => $boothNumber,
                'filename' => basename($results['warning_excel_path'])
            ]);
        }
        
        return response()->json([
            'success' => true,
            'message' => 'Batch import completed',
            'constituency' => $constituencyName,
            'booth_number' => $boothNumber,
            'directory' => $directory,
            'booth' => $results['booth'],
            'voters_files' => $results['voters_files'],
            'summary' => [
                'total_imported' => $results['total_imported'],
                'total_deleted' => $results['total_deleted'],
                'total_skipped' => $results['total_skipped'],
                'skipped_excel_url' => $skippedExcelUrl,
                'warning_excel_url' => $warningExcelUrl
            ],
            'skipped_voters' => $results['skipped_voters'] ?? [],
            'warning_data' => $results['warning_data'] ?? []
        ]);
    }

    /**
     * HTTP endpoint for processing all booths in a constituency (handles both encrypted and plain JSON requests)
     */
    public function runConstituency(Request $request)
    {
        // Try to get JSON data first (plain JSON request)
        $params = $request->json()->all();
        
        // If no JSON data, try decrypting (encrypted request)
        if (empty($params)) {
            $data = $request->getContent();
            $params = DecryptUtils::decryptRequestData($data);
        }
        
        // Validate we have data
        if (empty($params) || !is_array($params)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid request data',
                'error' => 'Request must contain valid JSON or encrypted data'
            ], 400);
        }
        
        // Call the actual import logic with decrypted data
        return $this->runConstituencyImport($params);
    }
    
    /**
     * Internal method for processing all booths in a constituency (accepts plain array)
     * Can be called from other controllers without encryption
     */
    public function runConstituencyImport(array $params)
    {
        $constituencyName = $params['constituency'] ?? null;
        
        // Validate required parameters
        if (!$constituencyName) {
            return response()->json([
                'success' => false,
                'message' => 'Constituency name is required',
                'error' => 'Please provide constituency parameter'
            ], 400);
        }
        
        // Construct constituency path
        // Use parent-level Constituency folder (shared across all client subdomains)
        $constituencyFolder = config('filesystems.disks.constituency.root');
        $constituencyPath = $constituencyFolder . '/' . $constituencyName;
        
        // Validate constituency folder exists
        if (!is_dir($constituencyPath)) {
            return response()->json([
                'success' => false,
                'message' => 'Constituency folder not found',
                'error' => "Constituency '{$constituencyName}' does not exist under Constituency folder",
                'path_checked' => $constituencyPath
            ], 404);
        }
        
        // Get all booth folders (numeric directories)
        $boothFolders = [];
        $items = scandir($constituencyPath);
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') continue;
            $itemPath = $constituencyPath . '/' . $item;
            if (is_dir($itemPath) && is_numeric($item)) {
                $boothFolders[] = $item;
            }
        }
        
        // Sort booth folders numerically
        sort($boothFolders, SORT_NUMERIC);
        
        if (empty($boothFolders)) {
            return response()->json([
                'success' => false,
                'message' => 'No booth folders found',
                'error' => "No numeric booth folders found in {$constituencyName}",
                'path_checked' => $constituencyPath
            ], 404);
        }
        
        // Process each booth
        $allResults = [];
        $totalImported = 0;
        $totalDeleted = 0;
        $totalSkipped = 0;
        $errors = [];
        
        $startIndex = (int)($params['start_index'] ?? 0);
        $maxImages = isset($params['max_images']) ? (int)$params['max_images'] : null;
        
        // Prepare import options using shared method
        $options = $this->prepareImportOptions($params);
        
        foreach ($boothFolders as $boothNumber) {
            try {
                $directory = $constituencyPath . '/' . $boothNumber;
                
                Log::info("Processing booth {$boothNumber} in constituency {$constituencyName}", [
                    'directory' => $directory
                ]);
                
                $job = new ProcessVoterImageBatch($directory, $startIndex, $maxImages, $options);
                $results = $job->handle();
                
                // Generate download URLs for Excel files if they exist
                $skippedExcelUrl = null;
                $warningExcelUrl = null;
                
                if (!empty($results['skipped_excel_path']) && file_exists($results['skipped_excel_path'])) {
                    $skippedExcelUrl = route('constituency.file', [
                        'constituency' => $constituencyName,
                        'booth' => $boothNumber,
                        'filename' => basename($results['skipped_excel_path'])
                    ]);
                }
                
                if (!empty($results['warning_excel_path']) && file_exists($results['warning_excel_path'])) {
                    $warningExcelUrl = route('constituency.file', [
                        'constituency' => $constituencyName,
                        'booth' => $boothNumber,
                        'filename' => basename($results['warning_excel_path'])
                    ]);
                }
                
                $allResults[$boothNumber] = [
                    'booth' => $results['booth'],
                    'summary' => [
                        'total_imported' => $results['total_imported'],
                        'total_deleted' => $results['total_deleted'],
                        'total_skipped' => $results['total_skipped'],
                        'skipped_excel_url' => $skippedExcelUrl,
                        'warning_excel_url' => $warningExcelUrl
                    ]
                ];
                
                $totalImported += $results['total_imported'];
                $totalDeleted += $results['total_deleted'];
                $totalSkipped += $results['total_skipped'];
                
            } catch (\Exception $e) {
                Log::error("Failed to process booth {$boothNumber}", [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                
                $errors[$boothNumber] = $e->getMessage();
            }
        }
        
        return response()->json([
            'success' => true,
            'message' => 'Constituency import completed',
            'constituency' => $constituencyName,
            'booths_processed' => count($allResults),
            'booths_failed' => count($errors),
            'total_summary' => [
                'total_imported' => $totalImported,
                'total_deleted' => $totalDeleted,
                'total_skipped' => $totalSkipped
            ],
            'booth_results' => $allResults,
            'errors' => $errors
        ]);
    }
}
