<?php

namespace App\Http\Controllers;

use App\Services\GiftService;
use Illuminate\Http\Request;
use App\Security\DecryptUtils;
use Illuminate\Http\Response as LaravelResponse;
use App\ResponseHandler\Response;

class GiftController extends Controller
{
    protected $giftService;
    protected $response;


    public function __construct(GiftService $giftService, Response $response)
    {
        $this->giftService = $giftService;
        $this->response = $response;
    }

    // Create Gift
    public function create(Request $request)
    {
        $data = $request->getContent();

        if (empty($data)) {
            return $this->response->respondWithError('Request data is empty');
        }

        // Step 2: Decrypt the incoming data
        $decryptedData = DecryptUtils::decryptRequestData($data);

        // Step 3: Ensure it’s a valid array
        if ($decryptedData == null || !is_array($decryptedData)) {
            return $this->response->respondWithError('Invalid or empty request data');
        }
        return $this->giftService->createGift($decryptedData);
    }

    // Get Gift by ID
    public function getById($id)
    {
        return $this->giftService->getGiftById($id);
    }

    // Get all Gifts
    public function getAll()
    {
        return $this->giftService->getAllGifts();
    }

    // Update Gift by ID
    public function update(Request $request, $id)
    {
        $data = $request->getContent();

        if (empty($data)) {
            return $this->response->respondWithError('Request data is empty');
        }

        // Step 2: Decrypt the incoming data
        $decryptedData = DecryptUtils::decryptRequestData($data);

        // Step 3: Ensure it’s a valid array
        if ($decryptedData == null || !is_array($decryptedData)) {
            return $this->response->respondWithError('Invalid or empty request data');
        }
        return $this->giftService->updateGiftById($id, $decryptedData);
    }

    // Soft delete Gift by ID
    public function delete($id)
    {
        return $this->giftService->deleteGiftById($id);
    }

    // Get all deleted Gifts
    public function getAllDeleted()
    {
        return $this->giftService->getAllDeletedGifts();
    }

    /**
     * Distribute gifts to ration cards
     * POST /api/gifts/distribute
     */
    public function distributeGift(Request $request)
    {
        try {

            $data = $request->getContent();

            if (empty($data)) {
                return $this->response->respondWithError('Request data is empty');
            }

            // Step 2: Decrypt the incoming data
            $decryptedData = DecryptUtils::decryptRequestData($data);

            // Step 3: Ensure it’s a valid array
            if ($decryptedData == null || !is_array($decryptedData)) {
                return $this->response->respondWithError('Invalid or empty request data');
            }
            return $this->giftService->distributeGift($decryptedData);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'Failed to distribute gifts: ' . $e->getMessage(),
                'data' => null
            ], 500);
        }
    }


  /**
     * Distribute gifts to ration cards
     * POST /api/gifts/distribute
     */
    public function distributeGiftBySingle(Request $request)
    {
        try {

            $data = $request->getContent();

            if (empty($data)) {
                return $this->response->respondWithError('Request data is empty');
            }

            // Step 2: Decrypt the incoming data
            $decryptedData = DecryptUtils::decryptRequestData($data);

            // Step 3: Ensure it’s a valid array
            if ($decryptedData == null || !is_array($decryptedData)) {
                return $this->response->respondWithError('Invalid or empty request data');
            }
            return $this->giftService->distributeSinleGift($decryptedData);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'Failed to distribute gifts: ' . $e->getMessage(),
                'data' => null
            ], 500);
        }
    }
    /**
     * Download gift distribution report as Excel
     * GET /api/gifts/{id}/export/gifted - Download gifted ration cards
     * GET /api/gifts/{id}/export/pending - Download pending ration cards
     */
    public function downloadGiftDistributionExcel($id, $gifted)
    {
        try {

            return $this->giftService->downloadGiftDistributionExcel($id, $gifted);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'Failed to generate Excel report: ' . $e->getMessage(),
                'data' => null
            ], 500);
        }
    }

    /**
     * Get gifts distributed to a specific ration card
     * GET /api/gifts/ration-card/{rationCardNumber}
     */
    public function getGiftsByRationCard($rationCardNumber)
    {
        return $this->giftService->getGiftsByRationCard($rationCardNumber);
    }
}
