# Voter Image Upload Solution

## The Problem
Voter images weren't being uploaded because the original voter creation endpoint (`/api/voters/create`) uses encrypted JSON data, which cannot handle file uploads. File uploads require `multipart/form-data` format.

## The Solution
I've implemented **two options** for handling voter creation with images:

### Option 1: Two-Step Process (Existing Endpoints)
1. **Create voter** using `/api/voters/create` (encrypted JSON)
2. **Upload image** using `/api/upload-voter-image` (multipart form)

### Option 2: Single Request (New Endpoint) ⭐ **RECOMMENDED**
Use the new `/api/voters/create-with-image` endpoint that handles both voter data and image upload in one request.

---

## New Endpoint: `/api/voters/create-with-image`

**Method:** `POST`
**Content-Type:** `multipart/form-data`
**URL:** `http://localhost/api/voters/create-with-image`

### Form Fields:
- `voter_id_number` (required) - string
- `name` (required) - string  
- `gender` (required) - Male|Female|Other
- `year_of_birth` (optional) - integer
- `date_of_birth` (optional) - YYYY-MM-DD format
- `anniversary_date` (optional) - YYYY-MM-DD format
- `is_head` (optional) - boolean
- `booth_id` (optional) - integer
- `booth_number` (optional) - string
- `mobile_number` (optional) - 10 digits
- `aadhar_number` (optional) - 12 digits
- `street_id` (optional) - integer
- `street_name` (optional) - string
- `ration_card_number` (optional) - string
- `voter_image` (optional) - image file (jpg,jpeg,png,gif, max 30MB)

### Example Using cURL:
```bash
curl -X POST "http://localhost/api/voters/create-with-image" \
  -F "voter_id_number=BCC1234567" \
  -F "name=RAJESWARI" \
  -F "gender=Male" \
  -F "year_of_birth=1989" \
  -F "date_of_birth=1989-06-09" \
  -F "anniversary_date=2017-06-26" \
  -F "is_head=false" \
  -F "booth_id=18" \
  -F "booth_number=B003" \
  -F "mobile_number=9698979194" \
  -F "aadhar_number=348234234234" \
  -F "street_id=10" \
  -F "street_name=Mettu street, vikravandi" \
  -F "voter_image=@/path/to/image.jpg"
```

### Example Using JavaScript/HTML Form:
```html
<form id="voterForm" enctype="multipart/form-data">
  <input type="text" name="voter_id_number" value="BCC1234567" required>
  <input type="text" name="name" value="RAJESWARI" required>
  <select name="gender" required>
    <option value="Male">Male</option>
    <option value="Female">Female</option>
    <option value="Other">Other</option>
  </select>
  <input type="number" name="year_of_birth" value="1989">
  <input type="date" name="date_of_birth" value="1989-06-09">
  <input type="date" name="anniversary_date" value="2017-06-26">
  <input type="checkbox" name="is_head" value="1">
  <input type="number" name="booth_id" value="18">
  <input type="text" name="booth_number" value="B003">
  <input type="tel" name="mobile_number" value="9698979194">
  <input type="text" name="aadhar_number" value="348234234234">
  <input type="number" name="street_id" value="10">
  <input type="text" name="street_name" value="Mettu street, vikravandi">
  <input type="file" name="voter_image" accept="image/*">
  <button type="submit">Create Voter</button>
</form>

<script>
document.getElementById('voterForm').addEventListener('submit', async (e) => {
  e.preventDefault();
  
  const formData = new FormData(e.target);
  
  try {
    const response = await fetch('/api/voters/create-with-image', {
      method: 'POST',
      body: formData
    });
    
    const result = await response.json();
    console.log('Success:', result);
  } catch (error) {
    console.error('Error:', error);
  }
});
</script>
```

### Success Response:
```json
{
  "status": 200,
  "message": "Voter created successfully",
  "data": {
    "id": 1,
    "voter_id_number": "BCC1234567",
    "name": "RAJESWARI",
    "gender": "Male",
    "year_of_birth": 1989,
    "date_of_birth": "1989-06-09",
    "anniversary_date": "2017-06-26",
    "is_head": false,
    "booth_id": 18,
    "booth_number": "B003",
    "mobile_number": "9698979194",
    "aadhar_number": "348234234234",
    "street_id": 10,
    "street_name": "Mettu street, vikravandi",
    "voter_image": "http://localhost/storage/voter_uploads/abc123.jpg",
    "ration_card_id": null,
    "created_at": "2025-11-18T11:30:00.000000Z",
    "updated_at": "2025-11-18T11:30:00.000000Z"
  }
}
```

---

## Key Features:
✅ **Single request** for voter creation + image upload
✅ **Automatic cleanup** - if voter creation fails, uploaded image is deleted
✅ **Proper validation** - form fields are validated before processing
✅ **Image optimization** - images stored in `storage/app/public/voter_uploads/`
✅ **Backward compatibility** - original endpoints still work
✅ **Proper error handling** - detailed error messages for troubleshooting

## Updated Voter Responses:
All voter retrieval endpoints now include the `voter_image` field with the full URL:
```json
"voter_image": "http://localhost/storage/voter_uploads/image_name.jpg"
```

---

## Alternative: Use Existing Two-Step Process

If you prefer to keep using the original encrypted JSON approach:

### Step 1: Create Voter (Original Method)
```bash
curl -X POST "http://localhost/api/voters/create" \
  -H "Content-Type: application/json" \
  -d '{encrypted_voter_data_here}'
```

### Step 2: Upload Image
```bash
curl -X POST "http://localhost/api/upload-voter-image" \
  -F "voter_id=1" \
  -F "image=@/path/to/image.jpg"
```

---

## Recommendation:
**Use the new `/api/voters/create-with-image` endpoint** for the best user experience as it handles everything in one request and provides better error handling.