# Multi-Tenant Constituency Setup - Quick Reference

## Current Setup (Development)

```
/Volumes/Workspace/Client Projects/
├── Constituency/                    ← Shared data (parent level)
│   ├── Lawspet/
│   │   ├── 1/
│   │   └── 2/
│   └── Orleanpet/
│       ├── 1/
│       └── 2/
└── ConstituencyApi/                 ← Your Laravel project
    └── .env → CONSTITUENCY_PATH=   (empty = auto-detect parent)
```

## Production Setup

### File Structure
```
/var/www/
├── Constituency/                           ← Shared data folder
│   ├── Lawspet/
│   ├── Orleanpet/
│   └── Kathirkamam/
├── lawspet-vaithianathan.ps25.in/         ← Client 1 subdomain
├── kathirkamam-vadivel.ps25.in/           ← Client 2 subdomain  
└── orleanpet-kumar.ps25.in/               ← Client 3 subdomain
```

### One-Time Setup Commands

```bash
# 1. Create shared Constituency folder
sudo mkdir -p /var/www/Constituency
sudo chown -R www-data:www-data /var/www/Constituency
sudo chmod -R 755 /var/www/Constituency

# 2. Upload constituency data
rsync -avz Constituency/ user@server:/var/www/Constituency/

# 3. For each client subdomain, add to .env:
echo "CONSTITUENCY_PATH=/var/www/Constituency" >> .env
```

## API Usage

### Single Booth Import
```bash
POST https://lawspet-vaithianathan.ps25.in/api/image-import/run
{
  "constituency": "Lawspet",
  "booth_number": "1"
}
```

### Full Constituency Import  
```bash
POST https://kathirkamam-vadivel.ps25.in/api/image-import/run-constituency
{
  "constituency": "Kathirkamam"
}
```

## How It Works

1. **Shared Data**: All client subdomains access `/var/www/Constituency`
2. **Configuration**: Each client's `.env` has `CONSTITUENCY_PATH=/var/www/Constituency`
3. **Automatic Fallback**: If `CONSTITUENCY_PATH` is empty, defaults to `../Constituency`
4. **Multi-Tenant Ready**: Add new clients without duplicating constituency data

## Benefits

✅ Single source of truth for constituency data  
✅ No data duplication across clients  
✅ Easy to update/add new constituencies  
✅ Reduced storage requirements  
✅ Centralized backup strategy

## Testing Commands

```bash
# Check path configuration
php artisan tinker --execute="echo config('filesystems.disks.constituency.root');"

# List available constituencies
ls -la /var/www/Constituency

# Test API call
curl -X POST http://localhost:8000/api/image-import/run \
  -H "Content-Type: application/json" \
  -d '{"constituency": "Lawspet", "booth_number": "1"}'
```

## Environment Variables

```env
# .env file for each client
CONSTITUENCY_PATH=/var/www/Constituency

# Leave empty for auto-detection (parent directory)
CONSTITUENCY_PATH=
```

## Nginx Configuration Template

```nginx
server {
    listen 80;
    server_name {CLIENT_SUBDOMAIN}.ps25.in;
    root /var/www/{CLIENT_SUBDOMAIN}.ps25.in/public;

    index index.php;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.2-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }
}
```

## Adding New Client Subdomain

```bash
# 1. Clone/deploy app to new subdomain folder
cd /var/www
git clone repo-url {new-client}.ps25.in

# 2. Configure .env
cd {new-client}.ps25.in
cp .env.example .env
echo "CONSTITUENCY_PATH=/var/www/Constituency" >> .env
php artisan key:generate

# 3. Set permissions
sudo chown -R www-data:www-data /var/www/{new-client}.ps25.in

# 4. Configure nginx
sudo nano /etc/nginx/sites-available/{new-client}.ps25.in
sudo ln -s /etc/nginx/sites-available/{new-client}.ps25.in /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl reload nginx

# 5. Test
curl https://{new-client}.ps25.in/api/image-import/run \
  -d '{"constituency": "Lawspet", "booth_number": "1"}'
```

## Troubleshooting

**Path not found?**
```bash
php artisan config:clear
php artisan cache:clear
```

**Permission denied?**
```bash
sudo chown -R www-data:www-data /var/www/Constituency
sudo chmod -R 755 /var/www/Constituency
```

**Check actual path used:**
```bash
php artisan tinker --execute="
echo 'Config Path: ' . config('filesystems.disks.constituency.root') . PHP_EOL;
echo 'Exists: ' . (is_dir(config('filesystems.disks.constituency.root')) ? 'Yes' : 'No');
"
```
