# Environment Configuration Guide

## Overview
This Laravel application supports multiple environments: **Development**, **Staging**, and **Production**. Each environment has its own configuration optimized for its specific use case.

## Environment Files

### 📁 Available Environment Files
- `.env.development` - Development environment
- `.env.staging` - Staging environment  
- `.env.production` - Production environment
- `.env` - Current active environment (automatically managed)

## 🔧 Environment Management

### Switch Between Environments
```bash
# Switch to development
./env-manager.sh development

# Switch to staging
./env-manager.sh staging

# Switch to production
./env-manager.sh production

# Check current environment
./env-manager.sh
```

### Manual Environment Switch
```bash
# Copy desired environment file to .env
cp .env.development .env

# Clear caches
php artisan optimize:clear
```

## 🚀 Deployment

### Development Deployment
```bash
./deployment/deploy-dev.sh
```
**Includes:**
- Development dependencies
- Database migrations
- Database seeding
- Debug mode enabled
- Local file storage

### Production Deployment
```bash
./deployment/deploy-prod.sh
```
**Includes:**
- Production dependencies only
- Database migrations
- Optimized caching
- Debug mode disabled
- Production security settings

## 📋 Environment Configurations

### Development Environment
- **APP_ENV**: `development`
- **APP_DEBUG**: `true`
- **Database**: Local MySQL/SQLite
- **Cache**: File-based
- **Queue**: Sync
- **Mail**: Local mail trap
- **Storage**: Local files
- **Rate Limit**: 1000 requests
- **CORS**: Allow all origins

### Staging Environment
- **APP_ENV**: `staging`
- **APP_DEBUG**: `true`
- **Database**: Staging MySQL server
- **Cache**: Redis
- **Queue**: Redis
- **Mail**: Mailtrap or staging SMTP
- **Storage**: AWS S3
- **Rate Limit**: 500 requests
- **CORS**: Staging domain only

### Production Environment
- **APP_ENV**: `production`
- **APP_DEBUG**: `false`
- **Database**: Production MySQL server
- **Cache**: Redis
- **Queue**: Redis
- **Mail**: Production SMTP (Gmail/SES)
- **Storage**: AWS S3
- **Rate Limit**: 100 requests
- **CORS**: Production domain only

## 🔐 Security Considerations

### Development
- Debug mode enabled for troubleshooting
- Relaxed CORS policies
- Local storage for easy development

### Production
- Debug mode disabled
- Strict CORS policies
- Secure headers enabled
- HTTPS enforcement
- Limited rate limiting
- Production-grade database

## 📝 Configuration Files to Modify

### 1. Environment Variables (`.env.*`)
Update these files with your specific settings:
- Database credentials
- Mail server settings
- AWS/S3 credentials
- Redis configuration
- API keys and secrets

### 2. Environment-Specific Logic (`config/environment.php`)
- API rate limits
- CORS origins
- Cache TTL values
- Debug settings

### 3. Application Configuration
- `config/cors.php` - CORS settings
- `config/database.php` - Database connections
- `config/cache.php` - Cache configuration
- `config/mail.php` - Mail configuration

## 🛠️ Usage in Code

### Using Environment Service
```php
use App\Services\EnvironmentService;

// Check environment
if (EnvironmentService::isDevelopment()) {
    // Development-specific code
}

// Get environment config
$rateLimit = EnvironmentService::getApiRateLimit();
$cacheTtl = EnvironmentService::getCacheTtl();
```

### Direct Environment Checks
```php
// Check environment
if (app()->environment('development')) {
    // Development code
}

// Get environment config
$config = config('environment.' . app()->environment());
```

## 🚨 Important Notes

1. **Never commit `.env`** - It should be in `.gitignore`
2. **Keep environment files secure** - They contain sensitive information
3. **Use different database names** for each environment
4. **Test deployments** in staging before production
5. **Monitor logs** in production environment
6. **Backup configurations** before making changes

## 🔄 CI/CD Integration

For automated deployments, you can use these scripts in your CI/CD pipeline:

```yaml
# Example GitHub Actions
- name: Deploy to Staging
  run: |
    ./env-manager.sh staging
    ./deployment/deploy-staging.sh

- name: Deploy to Production
  run: |
    ./env-manager.sh production
    ./deployment/deploy-prod.sh
```